
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Check, ShoppingCart, Briefcase, Building, GraduationCap, MessageCircle, Link as LinkIcon } from 'lucide-react';
import Image from 'next/image';
import type { Metadata } from 'next';
import Link from 'next/link';

export const metadata: Metadata = {
    title: 'Software Development Services | RiteClick Technology',
    description: 'Custom software development services including e-commerce sites, corporate websites, portfolio sites, school portals, and more. Build your digital presence with us.',
};

const services = [
  {
    title: 'E-commerce Websites',
    description: 'Powerful and scalable online stores with secure payment gateways, inventory management, and a seamless user experience to boost your sales.',
    icon: ShoppingCart,
    features: ['Custom storefront design', 'Secure payment integration', 'Product management', 'Mobile responsive', 'Mobile App Development'],
  },
  {
    title: 'Corporate Websites',
    description: 'Professional and polished websites that represent your brand identity, communicate your values, and build credibility with your audience.',
    icon: Building,
    features: ['Brand-centric design', 'Services showcase', 'Lead generation forms', 'Blog/News section'],
  },
  {
    title: 'Portfolio & Personal Websites',
    description: 'Showcase your work, skills, and personality with a unique and creative personal website that helps you stand out from the crowd.',
    icon: Briefcase,
    features: ['Unique personal branding', 'Project galleries', 'Contact forms', 'Resume/CV integration'],
  },
  {
    title: 'School & Educational Portals',
    description: 'Comprehensive portals for educational institutions with features for student management, course registration, online learning, and communication.',
    icon: GraduationCap,
    features: ['Student information system', 'Online course delivery', 'Result processing', 'Parent/Teacher communication'],
  },
];

export default function SoftwareDevelopmentPage() {
  return (
    <div className="flex flex-col min-h-screen">
      {/* Hero Section */}
      <section className="relative w-full py-20 md:py-32 lg:py-40 flex items-center justify-center text-center text-white">
        <Image
          src="/softwaredev.png"
          alt="Software Development"
          fill
          priority
          className="object-cover -z-10 brightness-50"
        />
        <div className="container px-4 md:px-6">
          <div className="flex flex-col items-center space-y-4">
            <h1 className="text-4xl font-bold tracking-tighter sm:text-5xl xl:text-6xl/none font-headline">
              Custom Software Development
            </h1>
            <p className="max-w-[700px] text-lg md:text-xl">
              Bringing your ideas to life with robust, scalable, and user-friendly web solutions tailored to your specific needs.
            </p>
          </div>
        </div>
      </section>

      {/* Services Section */}
      <section className="w-full py-12 md:py-24 lg:py-32 bg-secondary">
        <div className="container px-4 md:px-6">
          <div className="text-center">
            <h2 className="text-3xl font-bold tracking-tighter sm:text-5xl font-headline">What We Build</h2>
            <p className="max-w-[900px] mx-auto text-muted-foreground md:text-xl/relaxed mt-4">
              We specialize in developing a wide range of websites, from simple portfolios to complex enterprise applications.
            </p>
          </div>
          <div className="mx-auto grid grid-cols-1 gap-8 py-12 sm:grid-cols-2">
            {services.map((service, index) => {
              const Icon = service.icon;
              return (
                <Card key={index} className="flex flex-col transition-transform transform hover:-translate-y-2">
                  <CardHeader>
                    <div className="flex items-center gap-4">
                      <div className="bg-primary/10 rounded-full p-3 w-fit">
                        <Icon className="h-8 w-8 text-primary" />
                      </div>
                      <CardTitle className="font-headline text-2xl">{service.title}</CardTitle>
                    </div>
                  </CardHeader>
                  <CardContent className="flex-grow flex flex-col">
                    <p className="text-muted-foreground mb-6">{service.description}</p>
                    <ul className="space-y-3 flex-grow">
                      {service.features.map((feature, fIndex) => (
                         <li key={fIndex} className="flex items-start gap-3">
                            <Check className="h-5 w-5 text-primary mt-1 flex-shrink-0" />
                            <span className="text-muted-foreground">{feature}</span>
                        </li>
                      ))}
                    </ul>
                  </CardContent>
                </Card>
              );
            })}
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="w-full py-12 md:py-24 lg:py-32 bg-background">
        <div className="container text-center">
            <h2 className="text-3xl md:text-4xl font-bold font-headline">Have a Project in Mind?</h2>
            <p className="mt-4 max-w-2xl mx-auto text-muted-foreground">
              Let's discuss your requirements and how we can help you achieve your goals. Get in touch with us today for a free consultation.
            </p>
            <div className="mt-8 flex flex-col sm:flex-row items-center justify-center gap-4">
                <Button size="lg" asChild>
                  <Link href="https://calendly.com/riteclick/new-meeting" target="_blank" rel="noopener noreferrer">
                    Schedule a Free Consultation <LinkIcon className="ml-2" />
                  </Link>
                </Button>
                <Button size="lg" variant="outline" asChild>
                    <a href="https://wa.me/2349115572489" target="_blank" rel="noopener noreferrer">
                        <MessageCircle className="mr-2" /> Chat on WhatsApp
                    </a>
                </Button>
            </div>
        </div>
      </section>
    </div>
  );
}
