
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Check, BarChart3, Tags, MousePointerClick, Activity, MessageCircle, Link as LinkIcon } from 'lucide-react';
import Image from 'next/image';
import type { Metadata } from 'next';
import Link from 'next/link';

export const metadata: Metadata = {
    title: 'Digital Analytics Consultancy | RiteClick Technology',
    description: 'Expert digital analytics services to track, measure, and optimize your website performance. We specialize in Google Analytics, GTM, Hotjar, and Microsoft Clarity.',
};

const services = [
  {
    title: 'Web Analytics Implementation',
    description: 'We set up and configure powerful analytics tools like Google Analytics 4 to give you a comprehensive view of your website traffic and user engagement.',
    icon: BarChart3,
    features: ['Google Analytics 4 Setup', 'Cross-Domain Tracking', 'E-commerce Tracking', 'Custom Event Configuration'],
  },
  {
    title: 'Tag Management Solutions',
    description: 'Streamline your marketing and analytics tracking implementation with Google Tag Manager. We manage your tags efficiently for accurate data collection.',
    icon: Tags,
    features: ['Google Tag Manager Setup', 'Marketing Pixel Implementation', 'Custom Trigger Logic', 'Data Layer Development'],
  },
  {
    title: 'User Behavior Analysis',
    description: 'Understand how users interact with your site. We use tools like Hotjar and Microsoft Clarity to generate heatmaps, session recordings, and surveys.',
    icon: MousePointerClick,
    features: ['Heatmap & Scrollmap Analysis', 'User Session Recordings', 'On-site Surveys & Feedback', 'Conversion Funnel Visualization'],
  },
  {
    title: 'Performance Monitoring & Reporting',
    description: 'Stay on top of your website\'s performance with continuous monitoring and custom-built reports that deliver actionable insights directly to you.',
    icon: Activity,
    features: ['Custom Dashboard Creation', 'Automated Performance Reports', 'Anomaly Detection', 'Goal & KPI Tracking'],
  },
];

export default function DigitalAnalyticsConsultancyPage() {
  return (
    <div className="flex flex-col min-h-screen">
      {/* Hero Section */}
      <section className="relative w-full py-20 md:py-32 lg:py-40 flex items-center justify-center text-center text-white">
        <Image
          src="/DigitalAnalytics.png"
          alt="Digital Analytics Consultancy"
          fill
          priority
          className="object-cover -z-10 brightness-50"
        />
        <div className="container px-4 md:px-6">
          <div className="flex flex-col items-center space-y-4">
            <h1 className="text-4xl font-bold tracking-tighter sm:text-5xl xl:text-6xl/none font-headline">
              Digital Analytics Consultancy
            </h1>
            <p className="max-w-[700px] text-lg md:text-xl">
              Turn user behavior into actionable insights. We help you track, measure, and optimize your digital customer experience.
            </p>
          </div>
        </div>
      </section>

      {/* Services Section */}
      <section className="w-full py-12 md:py-24 lg:py-32 bg-secondary">
        <div className="container px-4 md:px-6">
          <div className="text-center">
            <h2 className="text-3xl font-bold tracking-tighter sm:text-5xl font-headline">Our Digital Analytics Services</h2>
            <p className="max-w-[900px] mx-auto text-muted-foreground md:text-xl/relaxed mt-4">
              We provide end-to-end analytics solutions to help you understand your users and improve your website's effectiveness.
            </p>
          </div>
          <div className="mx-auto grid grid-cols-1 gap-8 py-12 sm:grid-cols-2">
            {services.map((service, index) => {
              const Icon = service.icon;
              return (
                <Card key={index} className="flex flex-col transition-transform transform hover:-translate-y-2">
                  <CardHeader>
                    <div className="flex items-center gap-4">
                      <div className="bg-primary/10 rounded-full p-3 w-fit">
                        <Icon className="h-8 w-8 text-primary" />
                      </div>
                      <CardTitle className="font-headline text-2xl">{service.title}</CardTitle>
                    </div>
                  </CardHeader>
                  <CardContent className="flex-grow flex flex-col">
                    <p className="text-muted-foreground mb-6">{service.description}</p>
                    <ul className="space-y-3 flex-grow">
                      {service.features.map((feature, fIndex) => (
                         <li key={fIndex} className="flex items-start gap-3">
                            <Check className="h-5 w-5 text-primary mt-1 flex-shrink-0" />
                            <span className="text-muted-foreground">{feature}</span>
                        </li>
                      ))}
                    </ul>
                  </CardContent>
                </Card>
              );
            })}
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="w-full py-12 md:py-24 lg:py-32 bg-background">
        <div className="container text-center">
            <h2 className="text-3xl md:text-4xl font-bold font-headline">Ready to Optimize Your Website?</h2>
            <p className="mt-4 max-w-2xl mx-auto text-muted-foreground">
              Let's connect and discuss how our analytics experts can help you achieve your conversion goals. Get in touch for a free consultation.
            </p>
            <div className="mt-8 flex flex-col sm:flex-row items-center justify-center gap-4">
                <Button size="lg" asChild>
                  <Link href="https://calendly.com/riteclick/new-meeting" target="_blank" rel="noopener noreferrer">
                    Schedule a Free Consultation <LinkIcon className="ml-2" />
                  </Link>
                </Button>
                <Button size="lg" variant="outline" asChild>
                    <a href="https://wa.me/2349115572489" target="_blank" rel="noopener noreferrer">
                        <MessageCircle className="mr-2" /> Chat on WhatsApp
                    </a>
                </Button>
            </div>
        </div>
      </section>
    </div>
  );
}
